//
//  $Id: NSString_FJNStringEncodingUsingPerl.m 84 2008-03-18 13:25:05Z fujidana $
//
//  Copyright (c) 2006-2008 FUJIDANA. All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
// 3. Neither the name of the author nor the names of its contributors
//    may be used to endorse or promote products derived from this software
//    without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
// IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
// INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
// THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//

#import "NSString_FJNStringEncodingUsingPerl.h"


@implementation NSString (FJNStringEncodingUsingPerl)

+ (id)stringWithContentsOfFile:(NSString *)filepath withJcodePm:(NSString *)jcodePmPath usedEncoding:(NSStringEncoding *)enc error:(NSError **)error
{
	return [[[self alloc] initWithContentsOfFile:filepath
									 withJcodePm:jcodePmPath
									usedEncoding:enc
										   error:error] autorelease];
}

- (id)initWithContentsOfFile:(NSString *)filepath withJcodePm:(NSString *)jcodePmPath usedEncoding:(NSStringEncoding *)enc error:(NSError **)error
{
	NSString *code      = [NSString stringWithFormat:@"use Jcode; "
		@"open(INFILE, \"%@\"); "
		@"$content = join(\"\", <INFILE>); "
		@"close(INFILE); "
		@"$enc = Jcode::getcode($content); "
		@"print $enc;", 
		filepath];
	NSArray  *arguments = [NSArray arrayWithObjects:@"perl", @"-e", code, nil];
	NSTask   *task      = [[NSTask alloc] init];
	
	[task setLaunchPath:@"/usr/bin/env"];
	[task setArguments:arguments];
	if (jcodePmPath)
	{
		[task setCurrentDirectoryPath:[jcodePmPath stringByDeletingLastPathComponent]];
	}
	[task setStandardOutput:[NSPipe pipe]];
	[task setStandardError:[NSPipe pipe]];
	
	[task launch];
	[task waitUntilExit];
	
	NSData   *outputData   = [[[task standardOutput] fileHandleForReading] readDataToEndOfFile];
	NSString *outputString = [[[NSString alloc] initWithData:outputData encoding:NSASCIIStringEncoding] autorelease];
	NSData   *errorData    = [[[task standardOutput] fileHandleForReading] readDataToEndOfFile];
	NSString *errorString  = [[[NSString alloc] initWithData:errorData encoding:NSASCIIStringEncoding] autorelease];
	int      status        = [task terminationStatus];
	[task release];
	
	NSStringEncoding encoding;
	if ([outputString isEqualToString:@"ascii"])
	{
		encoding = NSASCIIStringEncoding;
	}
	else if ([outputString isEqualToString:@"euc"])
	{
		encoding = NSJapaneseEUCStringEncoding;
	}
	else if ([outputString isEqualToString:@"sjis"])
	{
		encoding = NSShiftJISStringEncoding;
	}
	else if ([outputString isEqualToString:@"jis"])
	{
		encoding = NSISO2022JPStringEncoding;
	}
	else if ([outputString isEqualToString:@"ucs2"])
	{
		encoding = NSUnicodeStringEncoding;
	}
	else if ([outputString isEqualToString:@"utf8"])
	{
		encoding = NSUTF8StringEncoding;
	}
	else
	{
		if (error != NULL)
		{
			NSDictionary *userInfo = nil;
			if (errorString && [errorString length] > 0)
			{
				userInfo = [NSDictionary dictionaryWithObject:errorString forKey:NSLocalizedDescriptionKey];
			}
			*error = [NSError errorWithDomain:NSPOSIXErrorDomain
										 code:status
									 userInfo:userInfo];
		}
		[self release];
		return nil;
	}
	
	self = [self initWithContentsOfFile:filepath
							   encoding:encoding
								  error:error];
	if (self != nil)
	{
		if (enc != NULL) *enc = encoding;
	}
	return self;
}

+ (id)stringWithContentsOfFile:(NSString *)path encodingGuessedByPerl:(NSStringEncoding *)enc error:(NSError **)error
{
	return [[[self alloc] initWithContentsOfFile:path
						   encodingGuessedByPerl:enc
										   error:error] autorelease];
}

- (id)initWithContentsOfFile:(NSString *)path encodingGuessedByPerl:(NSStringEncoding *)enc error:(NSError **)error
{
	NSString *code      = [NSString stringWithFormat:@"use Encode::Guess qw/euc-jp cp932 7bit-jis utf8/; "
		@"open(INFILE, \"%@\"); "
		@"my $content = join(\"\", <INFILE>); "
		@"close(INFILE); "
		@"my $enc = guess_encoding($content); "
		@"ref($enc) or die \"Can't guess: $enc\"; "
		@"print $enc->name;", 
		path];
	NSArray  *arguments = [NSArray arrayWithObjects:@"perl", @"-e", code, nil];
	NSTask   *task      = [[NSTask alloc] init];
	
	[task setLaunchPath:@"/usr/bin/env"];
	[task setArguments:arguments];
	[task setStandardOutput:[NSPipe pipe]];
	[task setStandardError:[NSPipe pipe]];
	
	[task launch];
	[task waitUntilExit];
	NSData   *outputData   = [[[task standardOutput] fileHandleForReading] readDataToEndOfFile];
	NSString *outputString = [[[NSString alloc] initWithData:outputData encoding:NSASCIIStringEncoding] autorelease];
	NSData   *errorData    = [[[task standardOutput] fileHandleForReading] readDataToEndOfFile];
	NSString *errorString  = [[[NSString alloc] initWithData:errorData encoding:NSASCIIStringEncoding] autorelease];
	int      status        = [task terminationStatus];
	[task release];
	
	NSStringEncoding encoding;
	if ([outputString isEqualToString:@"euc-jp"])
	{
		encoding = NSJapaneseEUCStringEncoding;
	}
	else if ([outputString isEqualToString:@"cp932"])
	{
		encoding = NSShiftJISStringEncoding;
	}
	else if ([outputString isEqualToString:@"7bit-jis"])
	{
		encoding = NSISO2022JPStringEncoding;
	}
	else if ([outputString isEqualToString:@"utf8"])
	{
		encoding = NSUTF8StringEncoding;
	}
	else
	{
		if (error != NULL)
		{
			NSDictionary *userInfo = nil;
			if (errorString && [errorString length] > 0)
			{
				userInfo = [NSDictionary dictionaryWithObject:errorString forKey:NSLocalizedDescriptionKey];
			}
			*error = [NSError errorWithDomain:NSPOSIXErrorDomain
										 code:status
									 userInfo:userInfo];
		}
		[self release];
		return nil;
	}
	
	self = [self initWithContentsOfFile:path 
							   encoding:encoding
								  error:error];
	if (self != nil)
	{
		if (enc != NULL) *enc = encoding;
	}
	return self;
}

@end
